/** @file   basecontroller.h
 * @brief   Declaration of BaseController - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:41 $
 * @author  Tomi Lamminsaari
 */
 
#ifndef H_WWW_BASECONTROLLER_H
#define H_WWW_BASECONTROLLER_H

#include "gameobject.h"

namespace WeWantWar {



/** @class  BaseController
 * @brief   Superclass for all the controller and AI-classes.
 * @author  Tomi Lamminsaari
 *
 * This class provides the interface the controller and ai-classes
 * must implement. So far there are only AI-classes that are derived from this
 * class.
 *
 * You must implement the <code>update()</code> - method before you
 * can create any instances of this class. To create an AI controller you must
 * not derive the your controller directly from this class but @ AIController
 * class instead.
 *
 * The controller's statuses can be accessed through the public members or
 * methods such as @c forward() or @c left()
 *
 */
class BaseController
{
public:
  ///
  /// Public members
  /// ==============
  
  /** Forward flag. Nonzero indicates forward movement. */
  int   m_forward;
  /** Backward flag. Nonzero indicates backward movement */
  int   m_backward;
  /** Left flag. Nonzero indicates movement to the left.  */
  int   m_left;
  /** Right flag. Nonzero indicates movement to the right. */
  int   m_right;
  /** The turning flag. Negative means rotation to the counter-clockwise and
   * positive to clockwise.
   */
  int   m_turn;
  /** The shoot flag. */
  int   m_shoot;
  

  ///
  /// Constructor, destructor
  /// =======================

  /** Constructor.
   * @param     pObject           Pointer to GameObject we're supposed to control.
   */
  BaseController( GameObject* pObject );
  
  /** Destructor
   */
  virtual ~BaseController();
  
  
  ///
  /// Methods
  /// =======
  
  /** Updates the controller commands. Must be implemented in the child
   * classes.
   */
  virtual void update() = 0;
  
  /** Sets the counter
   * @param     newCounter        New counter-value.
   */
  virtual void setCounter( int newCounter );
  
  /** Resets the controls.
   */
  virtual void reset();
  
  /** Returns a refence to @c forward - member so that in can be modified.
   * @return    A refence to forward - flag.
   */
  inline int& forward() { return m_forward; }
  
  /** Returns a refence to @c backward - flag so that in can be modified.
   * @return    A refence to backward - flag.
   */
  inline int& backward() { return m_backward; }
  
  /** Returns a refence to @c left - flag so that in can be modified.
   * @return    A refence to left - flag.
   */
  inline int& left() { return m_left; }
  
  /** Returns a refence to @c right - flag so that in can be modified.
   * @return    A refence to right - flag.
   */
  inline int& right() { return m_right; }
  
  /** Returns a refence to @c turn - flag so that in can be modified.
   * @return    A refence to turn - flag.
   */
  inline int& turn() { return m_turn; }
  
  /** Returns a refence to @c shoot - flag so that in can be modified.
   * @return    A refence to shoot - flag.
   */
  inline int& shoot() { return m_shoot; }
  
  
  
  ///
  /// Getter methods
  /// ==============
  

  /** Returns the counter-value
   * @return    Counter value
   */
  inline int counter() const { return m_counter; }
  
  /** Returns the pointer to the GameObject we're controlling.
   * @return    Pointer to the GameObject
   */
  inline GameObject* getGameObject() const { return m_pObject; }
  
  
  /** Returns the status of @c forward - flag.
   * @return    The forward - flag.
   */
  inline int forward() const { return m_forward; }
  
  /** Returns the status of @c backward - flag.
   * @return    The backward - flag.
   */
  inline int backward() const { return m_backward; }
  
  /** Returns the status of @c left - flag.
   * @return    The left - flag.
   */
  inline int left() const { return m_left; }
  
  /** Returns the status of @c right - flag.
   * @return    The right - flag.
   */
  inline int right() const { return m_right; }
  
  /** Returns the status of @c turn - flag.
   * @return    The turn - flag.
   */
  inline int turn() const { return m_turn; }
  
  /** Returns the status of @c shoot - flag.
   * @return    The shoot - flag.
   */
  inline int shoot() const { return m_shoot; }
  
protected:

  ///
  /// Members
  /// =======
  
  /** The counter */
  int   m_counter;
  
  /** Pointer to GameObject we're controlling. */
  GameObject* m_pObject;
  
private:
  BaseController( const BaseController& rO );
  BaseController& operator = ( const BaseController& rO );
};

};  // end of namespace

#endif // BaseController_H

/**
 * Version history
 * ===============
 * $Log: basecontroller.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:16:48+02  lamminsa
 * Initial revision
 *
 */
 
